/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <ntddk.h>
#include "mx_auto_config.h"
#include "mx_int.h"
#include "mcp_types.h"
#include "mx_ntddk.h"
#include "mx_arch_io.h"

/* For some reason, calling KeInitializeSemaphore in mx.c results in a
   link error. It may be a header issue (no function prototype?) so we
   wrap it here. Oddly enough, KeReleaseSemaphore is ok. */
void mx_init_sem(KSEMAPHORE* sem)
{
  KeInitializeSemaphore(sem, 0, 0x7fffffff);
}

int 
mx_pin_page(mx_instance_state_t *is, mx_page_pin_t *pin, int flags, uint64_t memory_context)
{
  PMDL mdl;
  PHYSICAL_ADDRESS pa;

  mdl = IoAllocateMdl((void*)(uintptr_t)pin->va, PAGE_SIZE, FALSE,
		      FALSE, NULL);
  if (mdl == NULL) {
    goto abort_with_nothing;
  }
  __try {
    MmProbeAndLockPages(mdl, KernelMode, IoModifyAccess);
    pa = MmGetPhysicalAddress((void*)(uintptr_t)pin->va);
    pin->dma.low = pa.LowPart;
    pin->dma.high = pa.HighPart;
    pin->mdl = mdl;
  }
  __except (EXCEPTION_EXECUTE_HANDLER) {
    goto abort_with_mdl;
  }
  return 0;

 abort_with_mdl:
  IoFreeMdl(mdl);
 abort_with_nothing:
  return EFAULT;
}

void
mx_unpin_page(mx_instance_state_t *is, mx_page_pin_t *pin, int flags)
{
  MmUnlockPages(pin->mdl);
  IoFreeMdl(pin->mdl);
}

#define MX_SCAN_PRIVILEGES 0
#if MX_SCAN_PRIVILEGES
#define MXSTR(x_) #x_
#define MX_CHECK(irp_, x_) \
do { \
  LUID luid; \
  BOOLEAN b; \
  luid = RtlConvertLongToLuid(x_); \
  b = SeSinglePrivilegeCheck(luid, irp_->RequestorMode); \
  DbgPrint(MXSTR(x_) " = %d\n", (int) b); \
} while(0)
#endif

int mx_can_be_privileged(PIRP irp)
{
  LUID lock_memory_luid;
  LUID load_driver_luid;
  LUID impersonate_luid;
  BOOLEAN b;

#if MX_SCAN_PRIVILEGES
  MX_CHECK(irp, SE_MIN_WELL_KNOWN_PRIVILEGE);
  MX_CHECK(irp, SE_CREATE_TOKEN_PRIVILEGE);
  MX_CHECK(irp, SE_ASSIGNPRIMARYTOKEN_PRIVILEGE);
  MX_CHECK(irp, SE_LOCK_MEMORY_PRIVILEGE);
  MX_CHECK(irp, SE_INCREASE_QUOTA_PRIVILEGE);

  MX_CHECK(irp, SE_MACHINE_ACCOUNT_PRIVILEGE);
  MX_CHECK(irp, SE_TCB_PRIVILEGE);
  MX_CHECK(irp, SE_SECURITY_PRIVILEGE);
  MX_CHECK(irp, SE_TAKE_OWNERSHIP_PRIVILEGE);
  MX_CHECK(irp, SE_LOAD_DRIVER_PRIVILEGE);
  MX_CHECK(irp, SE_SYSTEM_PROFILE_PRIVILEGE);
  MX_CHECK(irp, SE_SYSTEMTIME_PRIVILEGE);
  MX_CHECK(irp, SE_PROF_SINGLE_PROCESS_PRIVILEGE);
  MX_CHECK(irp, SE_INC_BASE_PRIORITY_PRIVILEGE);
  MX_CHECK(irp, SE_CREATE_PAGEFILE_PRIVILEGE);
  MX_CHECK(irp, SE_CREATE_PERMANENT_PRIVILEGE);
  MX_CHECK(irp, SE_BACKUP_PRIVILEGE);
  MX_CHECK(irp, SE_RESTORE_PRIVILEGE);
  MX_CHECK(irp, SE_SHUTDOWN_PRIVILEGE);
  MX_CHECK(irp, SE_DEBUG_PRIVILEGE);
  MX_CHECK(irp, SE_AUDIT_PRIVILEGE);
  MX_CHECK(irp, SE_SYSTEM_ENVIRONMENT_PRIVILEGE);
  MX_CHECK(irp, SE_CHANGE_NOTIFY_PRIVILEGE);
  MX_CHECK(irp, SE_REMOTE_SHUTDOWN_PRIVILEGE);
  MX_CHECK(irp, SE_UNDOCK_PRIVILEGE);
  MX_CHECK(irp, SE_SYNC_AGENT_PRIVILEGE);
  MX_CHECK(irp, SE_ENABLE_DELEGATION_PRIVILEGE);
  MX_CHECK(irp, SE_MANAGE_VOLUME_PRIVILEGE);
  MX_CHECK(irp, SE_IMPERSONATE_PRIVILEGE);
  /*MX_CHECK(irp, SE_CREATE_GLOBAL_PRIVILEGE);*/
#endif

  lock_memory_luid = RtlConvertLongToLuid(SE_LOCK_MEMORY_PRIVILEGE);
  load_driver_luid = RtlConvertLongToLuid(SE_LOAD_DRIVER_PRIVILEGE);
  impersonate_luid = RtlConvertLongToLuid(SE_IMPERSONATE_PRIVILEGE);
  b = (SeSinglePrivilegeCheck(lock_memory_luid, irp->RequestorMode) ||
       SeSinglePrivilegeCheck(load_driver_luid, irp->RequestorMode) ||
       SeSinglePrivilegeCheck(impersonate_luid, irp->RequestorMode));
  return (int)b;
}

#define MAX_PCI_BUSES 256

/* 0 - parent not found
 * 1 - parent found */
int
mx_find_parent_bridge(ULONG child_bus_number,
		      ULONG *parent_bus_number,
		      ULONG *parent_dev_number,
		      ULONG *parent_func_number)
{
  ULONG bus_number;
  ULONG dev_number;
  ULONG func_number;
  PCI_SLOT_NUMBER slot_number;
  PCI_COMMON_CONFIG pci_config;

  ULONG length;
  
  UCHAR secondary;
  UCHAR subordinate;

  bus_number = 0;
 again:
  for (; bus_number < MAX_PCI_BUSES; ++bus_number) {
    for (dev_number = 0; dev_number < PCI_MAX_DEVICES; ++dev_number) {
      slot_number.u.AsULONG=0;
      slot_number.u.bits.DeviceNumber = dev_number;
      for (func_number = 0; func_number < PCI_MAX_FUNCTION; ++func_number) {
        slot_number.u.bits.FunctionNumber = func_number;

	length = HalGetBusDataByOffset(PCIConfiguration, bus_number,
				       slot_number.u.AsULONG,
				       &pci_config, 0,
				       sizeof (pci_config));
	/* Devices can't have holes. If the device doesn't exist, move
	 * on to next bus. */
	if (length == 0) {
	  dev_number = PCI_MAX_DEVICES;
	  break;
	}
	/* Functions can have holes (or at least it's unspecified). If
	 * the function doesn't exist, move on to next function. */
	if (pci_config.VendorID == PCI_INVALID_VENDORID) {
	  continue;
	}

	/* 01h is a bridge. */
	if ((pci_config.HeaderType & 0x03) == 0x01) {
	  secondary = *((unsigned char*)&pci_config + 25);
	  subordinate = *((unsigned char*)&pci_config + 26);
	  if (secondary == child_bus_number) {
	    goto found;
	  }
	  else if (secondary < child_bus_number &&
		   child_bus_number <= subordinate) {
	    bus_number = secondary;
	    goto again;
	  }
	}
      }
    }
  }
  return 0;

 found:
  *parent_bus_number = bus_number;
  *parent_dev_number = dev_number;
  *parent_func_number = func_number;
  return 1;
}

int
mx_frob_chipsets(ULONG my_bus_number)
{
  ULONG bus_number;
  ULONG dev_number;
  ULONG func_number;
  int ret;
  PCI_SLOT_NUMBER slot_number;
  PCI_COMMON_CONFIG pci_config;
  PCI_COMMON_CONFIG *ppci_config;
  ULONG length;
  ULONG *ptr32;
  char *ptr;
  PHYSICAL_ADDRESS off;
  int enabled = 0;

  unsigned char *p;
  int i;

  ret = mx_find_parent_bridge(my_bus_number, &bus_number, &dev_number,
			      &func_number);
  if (ret == 1) {
    slot_number.u.AsULONG = 0;
    slot_number.u.bits.DeviceNumber = dev_number;
    slot_number.u.bits.FunctionNumber = func_number;
    length = HalGetBusDataByOffset(PCIConfiguration, bus_number,
				   slot_number.u.AsULONG,
				   &pci_config, 0,
				   sizeof (pci_config));
    if (length == 0) {
      goto done;
    }

#if 0
    DbgPrint("found device at bus %lu, device %lu, function %lu\n",
	     bus_number, dev_number, func_number);
    p = (unsigned char*)&pci_config;
    for (i = 0; i < 255; ++i) {
      DbgPrint("%02x ", *p++);
      if ((i % 16) == 15) {
	DbgPrint("\n");
      }
    }
    DbgPrint("\n");
#endif

    /* nVidia */
    if (pci_config.VendorID == 0x10de &&
	(pci_config.DeviceID == 0x005d ||
	 (pci_config.DeviceID >= 0x0374 &&
	  pci_config.DeviceID <= 0x0378))) {
      unsigned long mmconfig_base;

      if (pci_config.DeviceID == 0x005d)
	mmconfig_base = 0xe0000000UL;
      else 
	mmconfig_base = 0xf0000000UL;

      off.QuadPart = (mmconfig_base +
		      bus_number * 0x00100000UL +
		      (dev_number * 8 + func_number) * 0x00001000UL);

      ptr = MmMapIoSpace(off, 4096, MmNonCached);
      if (ptr != NULL) {
        ppci_config = (PCI_COMMON_CONFIG*)ptr;
	if (ppci_config->VendorID == pci_config.VendorID &&
	    ppci_config->DeviceID == pci_config.DeviceID) {
	  ptr32 = (ULONG*)(ptr + 0x178);
	  *ptr32 |= 0x40;
	  enabled = 1;
	}
	MmUnmapIoSpace(ptr, 4096);
      }
    }

    /* HT2000 */
    if (pci_config.VendorID == 0x1166 &&
	pci_config.DeviceID == 0x0132) {
	    enabled = 1;
    }
  }
 done:
  return enabled;
}

UCHAR mx_win_find_capability(UCHAR *pci_config, UCHAR cap_id)
{
  UCHAR cap;
  UCHAR ttl = 0;

  cap = pci_config[0x34];/* capability_list */
  while (cap && cap != 0xff && ttl++ < 64) {
    cap &= 0xfc;
    if (pci_config[cap] == cap_id)
      return cap;
    cap = pci_config[cap + 1];
  }
  return 0;
}

#define MX_PCI_CAP_ID_EXP 0x10
#define MX_PCI_EXP_LINK_CTL 0x10
#define MX_PCI_EXP_LINK_CTL_DISABLE 0x10

int
mx_frob_pcie_link(ULONG my_bus_number)
{
  ULONG bus_number;
  ULONG dev_number;
  ULONG func_number;
  int ret;
  PCI_SLOT_NUMBER slot_number;
  UCHAR pci_config[256];
  ULONG length;
  ULONG *ptr32;
  char *ptr;
  PHYSICAL_ADDRESS off;
  int enabled = 0;

  unsigned char *p;
  int i;

  ret = mx_find_parent_bridge(my_bus_number, &bus_number, &dev_number,
			      &func_number);
  if (ret == 1) {
    USHORT link_ctl;
    UCHAR exp_cap;

    slot_number.u.AsULONG = 0;
    slot_number.u.bits.DeviceNumber = dev_number;
    slot_number.u.bits.FunctionNumber = func_number;
    length = HalGetBusDataByOffset(PCIConfiguration, bus_number,
				   slot_number.u.AsULONG,
				   pci_config, 0,
				   256);
    if (length < 256)
      return -1;

    exp_cap = mx_win_find_capability(pci_config, MX_PCI_CAP_ID_EXP);
    if (!exp_cap)
      return -1;
    link_ctl = *(USHORT*)(pci_config + exp_cap + MX_PCI_EXP_LINK_CTL);
    link_ctl |= MX_PCI_EXP_LINK_CTL_DISABLE;
    HalSetBusDataByOffset(PCIConfiguration, bus_number,
			  slot_number.u.AsULONG,
			  &link_ctl, exp_cap + MX_PCI_EXP_LINK_CTL, 2);
    KeStallExecutionProcessor(100);
    link_ctl &= ~MX_PCI_EXP_LINK_CTL_DISABLE;
    HalSetBusDataByOffset(PCIConfiguration, bus_number,
			  slot_number.u.AsULONG,
			  &link_ctl, exp_cap + MX_PCI_EXP_LINK_CTL, 2);
    return 0;
  }
  return -1;
}

    
			  
    
    
